<?php

/**
 * Displays the size guide
 * @author jacek
 */
class ctSizeGuideDisplay
{

    protected $tables;
    protected $sg_id;

    /**
     * Initializes object
     */

    public function __construct()
    {
        add_action('woocommerce_before_single_product', array($this, 'displaySizeGuide'), 70);

        add_action('add_meta_boxes', array($this, 'chooseSizeGuide'));

        add_action('save_post', array($this, 'saveSizeGuideDropdown'));
        add_action('edit_post', array($this, 'saveSizeGuideDropdown'));

        add_shortcode('ct_size_guide', array($this, 'triggerSizeGuide'));

        add_action('wp_footer', array($this, 'overlayColor'));

        add_action('admin_init', array($this, 'wpmlRegister'));
    }

    /**
     * Registrate the custom wpml translations
     */
    public function wpmlRegister()
    {
        if (function_exists('icl_register_string')) {
            icl_register_string('plugin ct-size-guide', 'Button Label', $this->getSgOption('wc_size_guide_button_label'));
        }
    }

    /**
     * Return the WPML translation
     * @param $text
     * @return bool
     */
    public function wpmlTranslation($text, $source)
    {
        if (function_exists('icl_register_string')) {
            return icl_t('plugin ct-size-guide', 'Button Label', $text);
        }

        return $text;
    }

    /**
     * Size Guide MetaBox
     */

    public function chooseSizeGuide()
    {
        add_meta_box('ct_sizeguideopt', __('Choose size guide', 'ct_sgp'), array(
            $this,
            'renderSizeGuideDropdown'
        ), 'product', 'side');
    }

    /**
     * Select size guide per product
     *
     * @param $post
     */

    public function renderSizeGuideDropdown($post)
    {
        $args = array(
            'post_type' => 'ct_size_guide',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => "ASC"
        );

        $sg_list = get_posts($args);

        $current = get_current_screen()->action;

        //we have to check if there is already size guide post attached to the product
        $post_id = $post->ID;

        $sg_post_id = get_post_meta($post_id, '_ct_selectsizeguide');

        if ($sg_post_id) {
            $sg_post_id = $sg_post_id[0];
        }

        echo '<select id="ct_selectsizeguide" name="ct_selectsizeguide"><br>';
        echo '<option value="#NONE#">'.__('— Select —','ct_sgp') .'</option>';
        foreach ($sg_list as $sg_object) {
            $sg_id = $sg_object->ID;
            $sg_title = $sg_object->post_title;
            echo '<option value="' . $sg_id . '"' . selected($sg_post_id, $sg_id) . '>' . $sg_title . '</option>';
        }
        echo '</select>';

        echo '<br><br><input type="checkbox" name="ct_disablesizeguide" id="ct_disablesizeguide" ' . checked('checked', $sg_post_id, false) . '>' . __('Hide size guide from this product','ct_sgp').'</input>';
    }

    /**
     * Save size guide per product
     *
     * @param $post_id
     */

    public function saveSizeGuideDropdown($post_id)
    {
        $slug = 'product';
        // If this isn't a 'book' post, don't update it.
        if (!isset($_POST['post_type']) || $slug != $_POST['post_type']) {
            return;
        }

        if (isset($_POST['ct_selectsizeguide'])) {
            $disablesg = $_POST['ct_disablesizeguide'];
            if ($disablesg == 'on') {
                $disablesg = 'checked';
                update_post_meta($post_id, '_ct_selectsizeguide', $disablesg);
            } else {
                $selectedsg = $_POST['ct_selectsizeguide'];
                update_post_meta($post_id, '_ct_selectsizeguide', $selectedsg);

            }
        }
    }

    /**
     * Display size guide
     *
     * @param null $post_id
     */

    public function displaySizeGuide($post_id = null)
    {

        $post_id = $post_id ? $post_id : get_the_ID();
        $sg_post_id = get_post_meta($post_id, '_ct_selectsizeguide');
        if ($sg_post_id && $sg_post_id[0] != '#NONE#') {
            $sg_post_id = $sg_post_id[0];
            $this->sg_id = $sg_post_id;
            $size_table = get_post_meta($sg_post_id, '_ct_sizeguide');
            if ($size_table) {
                $size_table = $size_table[0];
                $this->tables = $size_table;
                $this->handleShortcodeRender($size_table, $sg_post_id);
            }
        } else {
            is_object($post_id) ? $post_id = $post_id->ID : '';

            $terms = wp_get_post_terms($post_id, 'product_cat');

            if (!empty($terms)) {
                $sg_post_id = get_woocommerce_term_meta($terms[0]->term_id, '_ct_assignsizeguide');
            }

            if ($sg_post_id) {
                $this->sg_id = $sg_post_id;
                $size_table = get_post_meta($sg_post_id, '_ct_sizeguide');
                if ($size_table) {
                    $size_table = $size_table[0];
                    $this->tables = $size_table;
                    $this->handleShortcodeRender($size_table, $sg_post_id);
                }
            }
        }
    }

    /**
     * Render shortcode
     */

    protected function handleShortcodeRender($size_table, $sg_post_id)
    {
        switch ($this->getSgOption("wc_size_guide_button_position")) {

            case 'ct-position-tab':
                add_filter('woocommerce_product_tabs', array($this, 'addSizeGuideTab'));
                break;

            case 'ct-position-add-to-cart':
                add_action('woocommerce_after_add_to_cart_button', array(
                    $this,
                    'doSgShortcode'
                ), $this->getSgOption('wc_size_guide_button_priority', 60));
                break;

            case 'ct-position-price':
                add_filter('woocommerce_get_price_html', array(
                    $this,
                    'addToSummary'
                ), $this->getSgOption('wc_size_guide_button_priority', 60));
                break;

            case 'ct-position-info':
                add_action('woocommerce_short_description', array(
                    $this,
                    'addToSummary'
                ), $this->getSgOption('wc_size_guide_button_priority', 60));
                break;
        }
        $this->renderSizeGuideTableOutput($size_table, $sg_post_id);
    }

    /**
     * Add Size Guide to product summary with info
     * @param $info
     * @return string
     */
    public function addToSummary($info)
    {
        return $info . do_shortcode('[ct_size_guide]');
    }

    /**
     * WooCommerce custom tab
     *
     * @param $tabs
     *
     * @return mixed
     */

    public function addSizeGuideTab($tabs)
    {
        $buttonLabel = $this->getSgOption("wc_size_guide_button_label", "Size Guide");

        $tabs['size_guide'] = array(
            'title' => $buttonLabel,
            'priority' => $this->getSgOption('wc_size_guide_button_priority', 20),
            'callback' => array($this, 'renderSizeGuideTab')
        );

        return $tabs;

    }

    /**
     * Render WooCommerce tab
     */

    public function renderSizeGuideTab()
    {
        $this->renderSizeGuideTableOutput($this->tables, $this->sg_id, true);
    }

    public function renderSizeGuideTableOutput($tables, $post_id, $is_tab = false)
    {
        $sg_object = get_post($post_id);
        $sg_title = $sg_object->post_title;
        $sg_content = $sg_object->post_content;
        if ($this->getSgOption('wc_size_guide_button_position', 'ct-position-summary') == 'ct-position-summary') {
            add_action('woocommerce_single_product_summary', array(
                $this,
                'doSgShortcode'
            ), $this->getSgOption('wc_size_guide_button_priority', 60));
        }
        if ($is_tab) {
            echo '<div id="ct_size_guide" class="sg ct_sg_tabbed">';
        } else {
            $pleft = $this->getSgOption('wc_size_guide_modal_padding_left', 0);
            $ptop = $this->getSgOption('wc_size_guide_modal_padding_top', 0);
            $pright = $this->getSgOption('wc_size_guide_modal_padding_right', 0);
            $pbottom = $this->getSgOption('wc_size_guide_modal_padding_bottom', 0);

            $paddings = '';

            if ($pleft > 0) {
                $paddings .= 'padding-left: ' . (int)$pleft . 'px; ';
            }
            if ($ptop > 0) {
                $paddings .= 'padding-top: ' . (int)$ptop . 'px; ';
            }
            if ($pright > 0) {
                $paddings .= 'padding-right: ' . (int)$pright . 'px; ';
            }
            if ($pbottom > 0) {
                $paddings .= 'padding-bottom: ' . (int)$pbottom . 'px; ';
            }
            echo '<div id="ct_size_guide" style="' . $paddings . '" class="sg mfp-hide">';
        }

        echo '<h1 class="sg_title">' . $sg_title . '</h1><hr>';
        echo '<div class="sg_content">';
        print_r($sg_content);
        echo '</div>';
        echo '<div style="clear:both;"></div>';
        echo '<hr>';
        $c = count($tables);

        foreach ($tables as $key => $table) {
            if ($table['title']) {
                echo '<h4 class="ct_table_title">' . do_shortcode($table['title']) . '</h4>';
            }

            echo '<table>';
            $row_mark = 1;

            foreach ($table['table'] as $row) {
                $col_mark = 1;
                echo '<tr>';
                foreach ($row as $cell) {
                    if ($row_mark == 1 || $col_mark == 1) {
                        echo '<th>' . $cell . '</th>';
                    } else {
                        echo '<td>' . $cell . '</td>';
                    }
                    $col_mark++;
                }
                $row_mark++;
                echo '</tr>';
            }
            echo '</table>';
            if ($table['caption']) {
                echo '<br><p class="ct_table_caption">' . do_shortcode($table['caption']) . '</p>';
            }
            if ($key < $c - 1) {
                echo '<hr>';
            }
        }
        echo '</div>';

    }

    public function triggerSizeGuide()
    {

        global $product;
        $productStock = $product->get_stock_quantity();
        $productAvailability = $product->get_availability();
        $hide = get_option('wc_size_guide_hide');


        if ($hide != 'yes' && $productAvailability['availability'] != 'Out of stock') {
            $trigger = $this->getSgOption('wc_size_guide_button_style', 'ct-trigger-button');


            $align = $this->getSgOption('wc_size_guide_button_align', 'left');
            if ($this->getSgOption('wc_size_guide_button_position') == 'ct-position-add-to-cart') {
                $align = '';
            }
            $clear = $this->getSgOption('wc_size_guide_button_clear', 'no');

            $mleft = $this->getSgOption('wc_size_guide_button_margin_left', 0);
            $mtop = $this->getSgOption('wc_size_guide_button_margin_top', 0);
            $mright = $this->getSgOption('wc_size_guide_button_margin_right', 0);
            $mbottom = $this->getSgOption('wc_size_guide_button_margin_bottom', 0);

            $margins = '';

            if ($mleft != 0) {
                $margins .= 'margin-left: ' . (int)$mleft . 'px; ';
            }
            if ($mtop != 0) {
                $margins .= 'margin-top: ' . (int)$mtop . 'px; ';
            }
            if ($mright != 0) {
                $margins .= 'margin-right: ' . (int)$mright . 'px; ';
            }
            if ($mbottom != 0) {
                $margins .= 'margin-bottom: ' . (int)$mbottom . 'px; ';
            }


            $buttonLabel = $this->getSgOption("wc_size_guide_button_label", "Size Guide");

            if ($trigger == 'ct-trigger-button') {
                $output = '<a class="open-popup-link ' . $this->getSgOption('wc_size_guide_button_class', 'button_sg') . '" href="#ct_size_guide" style="float: ' . $align . '; ' . $margins . '">' . $buttonLabel . '</a>';
            } else {
                $output = '<a class="open-popup-link" href="#ct_size_guide" style="float: ' . $align . '; ' . $margins . '">' . $buttonLabel . '</a>';
            }
            if ($clear == 'no') {
                $output .= '<div class="clearfix"></div>';
            }
        }
        return $output;
    }

    protected function getSgOption($opt, $default = "null")
    {

        $val = get_post_meta($this->sg_id, '_ct_sizeguidesettings');
        if ($val) {
            $val = $val[0];
        }

        if (isset($val[$opt]) && $val[$opt] == 'global' || !$val) {
            $val = get_option($opt, $default);
        } elseif (isset($val[$opt])) {
            $val = $val[$opt];
        } else {
            $val = $default;
        }

        return $val;

    }

    public function overlayColor()
    {
        echo '<style>.mfp-bg{background:' . $this->getSgOption('wc_size_guide_overlay_color', '#000000') . ';}</style>';
    }

    public function doSgShortcode()
    {
        echo do_shortcode('[ct_size_guide]');
    }

//    public function wpmlTranslation($text)
//    {
//        if (function_exists('icl_register_string')) {
//            icl_register_string('plugin ct-size-guide', 'Button Label', $text);
//
//            return icl_t('plugin ct-size-guide', 'Button Label', $text);
//        }
//
//        return $text;
//    }

}

new ctSizeGuideDisplay();